<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Chat Application</title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome for icons -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">

    <style>
        :root {
    --primary-color: #007bff;
    --secondary-color: #6c757d;
    --light-color: #f8f9fa;
    --dark-color: #343a40;
    --user-message-bg: #e3f2fd;
    --bot-message-bg: #f8f9fa;
    --sidebar-width: 300px;
}

body {
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    height: 100vh;
    background-color: #f5f5f5;
    margin: 0;
    padding: 0;
}

.chat-container {
    height: 100vh;
    padding: 0;
}

.sidebar {
    background-color: white;
    border-right: 1px solid #e0e0e0;
    height: 100vh;
    overflow-y: auto;
    padding: 0;
}

.sidebar-header {
    padding: 15px;
    border-bottom: 1px solid #e0e0e0;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.chat-list {
    padding: 10px 0;
}

.chat-item {
    padding: 10px 15px;
    cursor: pointer;
    border-bottom: 1px solid #f0f0f0;
    transition: background-color 0.2s;
}

.chat-item:hover {
    background-color: #f5f5f5;
}

.chat-item.active {
    background-color: var(--user-message-bg);
}

.chat-item-title {
    font-weight: 500;
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.chat-item-preview {
    font-size: 0.8rem;
    color: var(--secondary-color);
    white-space: nowrap;
    overflow: hidden;
    text-overflow: ellipsis;
}

.main-chat-area {
    display: flex;
    flex-direction: column;
    height: 100vh;
    background-color: white;
    position: relative;
}

.chat-header {
    padding: 15px;
    border-bottom: 1px solid #e0e0e0;
    display: flex;
    align-items: center;
    gap: 10px;
}

.messages-container {
    flex: 1;
    overflow-y: auto;
    padding: 20px;
    background-color: #f9f9f9;
}

.welcome-message {
    text-align: center;
    margin-top: 50px;
    color: var(--secondary-color);
}

.message {
    margin-bottom: 15px;
    max-width: 80%;
}

.message-content {
    padding: 10px 15px;
    border-radius: 18px;
    word-wrap: break-word;
}

.user-message {
    margin-left: auto;
}

.user-message .message-content {
    background-color: var(--user-message-bg);
    border-bottom-right-radius: 5px;
}

.bot-message {
    margin-right: auto;
}

.bot-message .message-content {
    background-color: var(--bot-message-bg);
    border-bottom-left-radius: 5px;
}

.message-time {
    font-size: 0.7rem;
    color: var(--secondary-color);
    margin-top: 5px;
}

.input-area {
    padding: 15px;
    border-top: 1px solid #e0e0e0;
    background-color: white;
}

.input-group {
    position: relative;
}

#message-input {
    resize: none;
    border-radius: 20px;
    padding-right: 45px;
    max-height: 120px;
    overflow-y: auto;
}

#send-btn {
    position: absolute;
    right: 5px;
    bottom: 5px;
    border-radius: 50%;
    width: 36px;
    height: 36px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.typing-indicator {
    display: flex;
    align-items: center;
    gap: 5px;
    margin-top: 5px;
    font-size: 0.8rem;
    color: var(--secondary-color);
}

.typing-dots {
    display: flex;
    gap: 3px;
}

.typing-dots span {
    width: 6px;
    height: 6px;
    background-color: var(--secondary-color);
    border-radius: 50%;
    animation: typingAnimation 1.4s infinite ease-in-out;
}

.typing-dots span:nth-child(1) {
    animation-delay: 0s;
}

.typing-dots span:nth-child(2) {
    animation-delay: 0.2s;
}

.typing-dots span:nth-child(3) {
    animation-delay: 0.4s;
}

@keyframes typingAnimation {
    0%, 60%, 100% {
        transform: translateY(0);
    }
    30% {
        transform: translateY(-5px);
    }
}

/* Responsive adjustments */
@media (max-width: 768px) {
    .sidebar {
        position: absolute;
        z-index: 1000;
        width: var(--sidebar-width);
        height: 100vh;
        transform: translateX(-100%);
        transition: transform 0.3s ease;
    }
    
    .sidebar.show {
        transform: translateX(0);
    }
    
    .main-chat-area {
        width: 100%;
    }
}
    </style>
</head>
<body>
    <div class="container-fluid chat-container">
        <div class="row h-100">
            <!-- Sidebar -->
            <div class="col-md-3 sidebar d-none d-md-block">
                <div class="sidebar-header">
                    <h4>Chats</h4>
                    <button class="btn btn-sm btn-primary" id="new-chat-btn">
                        <i class="fas fa-plus"></i> New Chat
                    </button>
                </div>
                <div class="chat-list" id="chat-list">
                    <!-- Chat history items will be added here dynamically -->
                </div>
            </div>
            
            <!-- Main Chat Area -->
            <div class="col-md-9 main-chat-area">
                <div class="chat-header">
                    <button class="btn btn-sm d-md-none" id="sidebar-toggle">
                        <i class="fas fa-bars"></i>
                    </button>
                    <h4 id="current-chat-title">New Chat</h4>
                </div>
                
                <div class="messages-container" id="messages-container">
                    <!-- Messages will be added here dynamically -->
                    <div class="welcome-message">
                        <h3>Welcome to Chat</h3>
                        <p>Start a new conversation or select an existing one from the sidebar.</p>
                    </div>
                </div>
                
                <div class="input-area">
                    <div class="input-group">
                        <textarea class="form-control" id="message-input" placeholder="Type your message here..." rows="1"></textarea>
                        <button class="btn btn-primary" id="send-btn">
                            <i class="fas fa-paper-plane"></i>
                        </button>
                    </div>
                    <div class="typing-indicator d-none" id="typing-indicator">
                        <span>AI is typing</span>
                        <div class="typing-dots">
                            <span></span>
                            <span></span>
                            <span></span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Bootstrap JS Bundle with Popper -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <!-- Custom JS -->
    <script >
        document.addEventListener('DOMContentLoaded', function() {
    // DOM Elements
    const messageInput = document.getElementById('message-input');
    const sendBtn = document.getElementById('send-btn');
    const messagesContainer = document.getElementById('messages-container');
    const chatList = document.getElementById('chat-list');
    const newChatBtn = document.getElementById('new-chat-btn');
    const currentChatTitle = document.getElementById('current-chat-title');
    const typingIndicator = document.getElementById('typing-indicator');
    const sidebarToggle = document.getElementById('sidebar-toggle');
    const sidebar = document.querySelector('.sidebar');
    
    // State
    let currentChatId = null;
    let chats = [];
    
    // Initialize
    loadChatHistory();
    setupEventListeners();
    
    function setupEventListeners() {
        // Send message on button click
        sendBtn.addEventListener('click', sendMessage);
        
        // Send message on Enter key (but allow Shift+Enter for new lines)
        messageInput.addEventListener('keydown', function(e) {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                sendMessage();
            }
            
            // Auto-resize textarea as user types
            this.style.height = 'auto';
            this.style.height = (this.scrollHeight) + 'px';
        });
        
        // New chat button
        newChatBtn.addEventListener('click', createNewChat);
        
        // Sidebar toggle for mobile
        sidebarToggle.addEventListener('click', function() {
            sidebar.classList.toggle('show');
        });
        
        // Click outside sidebar to close it on mobile
        document.addEventListener('click', function(e) {
            if (window.innerWidth <= 768 && !sidebar.contains(e.target) && 
                e.target !== sidebarToggle && !sidebarToggle.contains(e.target)) {
                sidebar.classList.remove('show');
            }
        });
    }
    
    function sendMessage() {
        const messageText = messageInput.value.trim();
        if (!messageText) return;
        
        // Create new chat if this is the first message
        if (!currentChatId) {
            createNewChat();
        }
        
        // Add user message to UI
        addMessageToUI('user', messageText);
        messageInput.value = '';
        messageInput.style.height = 'auto';
        
        // Show typing indicator
        showTypingIndicator(true);
        
        // Here you would typically call your API
        // For now, we'll simulate a response after a delay
        simulateApiResponse(messageText);
    }
    
    function addMessageToUI(sender, text) {
        const messageDiv = document.createElement('div');
        messageDiv.className = `message ${sender}-message`;
        
        const messageContent = document.createElement('div');
        messageContent.className = 'message-content';
        messageContent.textContent = text;
        
        const messageTime = document.createElement('div');
        messageTime.className = 'message-time';
        messageTime.textContent = new Date().toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
        
        messageDiv.appendChild(messageContent);
        messageDiv.appendChild(messageTime);
        
        messagesContainer.appendChild(messageDiv);
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
        
        // Update chat preview in sidebar
        if (sender === 'user') {
            updateChatPreview(currentChatId, text);
        }
    }
    
    function showTypingIndicator(show) {
        if (show) {
            typingIndicator.classList.remove('d-none');
        } else {
            typingIndicator.classList.add('d-none');
        }
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
    }
    
    function simulateApiResponse(userMessage) {
        // Simulate API delay
        setTimeout(() => {
            showTypingIndicator(false);
            
            // Generate a simple response (in a real app, this would come from your API)
            let responseText = "I'm a simulated response to: " + userMessage;
            
            // Add bot message to UI
            addMessageToUI('bot', responseText);
        }, 1500);
    }
    
    function createNewChat() {
        // Generate a new chat ID
        const newChatId = 'chat-' + Date.now();
        const newChatTitle = 'Chat ' + (chats.length + 1);
        
        // Create chat object
        const newChat = {
            id: newChatId,
            title: newChatTitle,
            lastMessage: '',
            timestamp: Date.now()
        };
        
        // Add to chats array
        chats.unshift(newChat);
        saveChatsToLocalStorage();
        
        // Update UI
        addChatToSidebar(newChat);
        switchToChat(newChatId);
        
        // Clear messages container and show welcome message
        messagesContainer.innerHTML = `
            <div class="welcome-message">
                <h3>Welcome to Chat</h3>
                <p>Start a new conversation with the AI assistant.</p>
            </div>
        `;
    }
    
    function addChatToSidebar(chat) {
        const chatItem = document.createElement('div');
        chatItem.className = 'chat-item';
        chatItem.dataset.chatId = chat.id;
        
        chatItem.innerHTML = `
            <div class="chat-item-title">${chat.title}</div>
            <div class="chat-item-preview">${chat.lastMessage || 'No messages yet'}</div>
        `;
        
        chatItem.addEventListener('click', function() {
            switchToChat(chat.id);
            if (window.innerWidth <= 768) {
                sidebar.classList.remove('show');
            }
        });
        
        chatList.insertBefore(chatItem, chatList.firstChild);
    }
    
    function switchToChat(chatId) {
        // Update current chat ID
        currentChatId = chatId;
        
        // Update active state in sidebar
        document.querySelectorAll('.chat-item').forEach(item => {
            if (item.dataset.chatId === chatId) {
                item.classList.add('active');
                currentChatTitle.textContent = item.querySelector('.chat-item-title').textContent;
            } else {
                item.classList.remove('active');
            }
        });
        
        // In a real app, you would load the chat history from your API here
        // For now, we'll just show a placeholder
        messagesContainer.innerHTML = `
            <div class="welcome-message">
                <h3>${currentChatTitle.textContent}</h3>
                <p>This is where your conversation would appear.</p>
            </div>
        `;
    }
    
    function updateChatPreview(chatId, lastMessage) {
        const chat = chats.find(c => c.id === chatId);
        if (chat) {
            chat.lastMessage = lastMessage;
            chat.timestamp = Date.now();
            saveChatsToLocalStorage();
            
            // Update sidebar UI
            const chatItem = document.querySelector(`.chat-item[data-chat-id="${chatId}"]`);
            if (chatItem) {
                chatItem.querySelector('.chat-item-preview').textContent = lastMessage;
                
                // Move to top of list
                chatList.insertBefore(chatItem, chatList.firstChild);
            }
        }
    }
    
    function loadChatHistory() {
        // In a real app, you would load this from your API
        // For now, we'll use localStorage for demo purposes
        const savedChats = localStorage.getItem('chatHistory');
        if (savedChats) {
            chats = JSON.parse(savedChats);
            
            // Add chats to sidebar
            chats.forEach(chat => {
                addChatToSidebar(chat);
            });
            
            // If there are chats, select the most recent one
            if (chats.length > 0) {
                switchToChat(chats[0].id);
            }
        }
    }
    
    function saveChatsToLocalStorage() {
        localStorage.setItem('chatHistory', JSON.stringify(chats));
    }
});
    </script>
</body>
</html>